<?php
/*--------------------------------------------------------------
   Writer.php 2021-06-17
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Data;

use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDatabaseEntry;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Height;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\HorizontalPosition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Id;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Label;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Option\UpdateSet;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\VerticalPosition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Width;
use Gambio\Admin\Modules\Statistics\App\Overview\SchemaValidator;

class Writer
{
    private const SCHEMA_ID = "https://gambio.com/schema/DatabaseEntry.json";
    
    /**
     * @var Database
     */
    private $database;
    
    /**
     * @var SchemaValidator
     */
    private $schemaValidator;
    
    
    /**
     * Constructor.
     */
    public function __construct(Database $database, SchemaValidator $schemaValidator)
    {
        $this->database        = $database;
        $this->schemaValidator = $schemaValidator;
    }
    
    
    /**
     * Add widget.
     */
    public function addWidget(WidgetDatabaseEntry $databaseEntry): void
    {
        $data = [
            "type"               => $databaseEntry->type()->value(),
            "label"              => $databaseEntry->label()->value(),
            "width"              => $databaseEntry->width()->value(),
            "height"             => $databaseEntry->height()->value(),
            "horizontalPosition" => $databaseEntry->horizontalPosition()->value(),
            "verticalPosition"   => $databaseEntry->verticalPosition()->value(),
            "options"            => $databaseEntry->updateSet()->toArray(),
        ];
        
        $this->schemaValidator->validate(self::SCHEMA_ID, $data);
        $this->database->updateOrInsert($data);
    }
    
    
    /**
     * Update widget dimension.
     */
    public function resizeWidget(Id $id, Width $width, Height $height): void
    {
        $this->database->updateById($id->value(),
                                    [
                                        "width"  => $width->value(),
                                        "height" => $height->value(),
                                    ]);
    }
    
    
    /**
     * Update widget position.
     */
    public function moveWidget(Id $id, HorizontalPosition $horizontalPosition, VerticalPosition $verticalPosition): void
    {
        $this->database->updateById($id->value(),
                                    [
                                        "horizontalPosition" => $horizontalPosition->value(),
                                        "verticalPosition"   => $verticalPosition->value(),
                                    ]);
    }
    
    
    /**
     * Update widget options.
     */
    public function configureWidget(Id $id, UpdateSet $updateSet): void
    {
        $options = [];
        
        foreach ($updateSet->getIterator() as $optionId => $optionValue) {
            $options["options.$optionId"] = $optionValue;
        }
        
        $this->database->updateById($id->value(), $options);
    }
    
    
    /**
     * Delete widget.
     */
    public function deleteWidget(Id $id): void
    {
        $this->database->deleteById($id->value());
    }
    
    
    /**
     * Rename widget.
     */
    public function renameWidget(Id $id, Label $label): void
    {
        $this->database->updateById($id->value(), ["label" => $label->value()]);
    }
}